### RDP GRAPH GENERATOR MASTER ### 
# This script contains, in order, all the code to generate 
# all the graphs from the 2022 CPS RDP (where R was used to generate the graphs).
# The script does not contain code to generate Figure 5: Trends in Card and Cash Payments
# and Figure 25: Contactless Card Payments since these were prepared using non-CPS data.
# Authors: Tanya Livermore, Jack Mulqueeney, Thuong Nguyen, Benjamin Watson
# Date: 14/11/2023


# Load required packages
library(tidyverse)
library(readstata13)

# Read past years' data
data13 <- read.dta13(".../2013 Consumer Use Study - Finalised Dataset.dta",
                     generate.factors = T, nonint.factors = T)
data16 <- read.dta13(".../2016 CPS - Finalised Dataset (main only).dta", 
                     generate.factors = T, nonint.factors = T)
data19 <- read.dta13(".../2019 CPS - Finalised Dataset.dta",
                     generate.factors = T, convert.factors = T)

### Section 3: Key Trends in Consumer Payments ###
# Figure 1: Cash and Card Payments #

fig1_left_panel <- diary_full %>% 
  filter(paymeth_crosswave != "Missing", paymeth_crosswave !="Transport card", !is.na(paymeth_crosswave), payamt < 9999, payamt >= 0) %>% 
  filter(!(paypurp_crosswave %in% c("Cash deposit", "Transfers to a family member or friend", "Transfer within own accounts"))) %>% 
  group_by(Survey_year, paymethcons2) %>% 
  mutate(counttrans = 1) %>% 
  summarise(number = sum(dweight*counttrans)) %>% 
  group_by(Survey_year) %>% 
  mutate(number = number*100/sum(number)) %>% 
  filter(paymethcons2 %in% c("Cash", "Card")) %>% 
  ungroup()

fig1_right_panel <- diary_full %>% 
  filter(paymeth_crosswave != "Missing", paymeth_crosswave !="Transport card", !is.na(paymeth_crosswave), payamt < 9999, payamt >= 0) %>% 
  filter(!(paypurp_crosswave %in% c("Cash deposit", "Transfers to a family member or friend", "Transfer within own accounts"))) %>% 
  group_by(Survey_year, paymethcons3) %>% 
  mutate(counttrans = 1) %>% 
  summarise(number = sum(dweight*counttrans)) %>% 
  group_by(Survey_year) %>% 
  mutate(number = number*100/sum(number)) %>% 
  filter(paymethcons3 %in% c("Debit", "Credit")) %>% 
  ungroup()


# Figure 2: Use of One Payment Method Only #
fig2 <- diary_full %>% 
  mutate(paychan_crosswave = ifelse(Survey_year == 2019 & paychan2019 %in% c("Tap card", "Other in person", "Insert card", "Tap device"), "In person", paychan_crosswave)) %>% 
  filter(paychan_crosswave == "In person", paymeth_crosswave != "Missing", !is.na(paymeth_crosswave), paymeth_crosswave != "NA", payamt >= 0, payamt < 9999) %>% 
  filter(!(paypurp_crosswave %in% c("Cash deposit", "Transfers to a family member or friend", "Transfer within own accounts"))) %>% 
  mutate(idtag_inperson = ifelse(lag(Respondent_ID, 1, default = 1) == Respondent_ID, 0, 1)) %>% 
  group_by(Survey_year) %>% 
  mutate(total_n = sum(idtag_inperson*dweight)) %>% 
  select(Respondent_ID, Survey_year, total_n, paymethcons, dweight) %>% 
  group_by(Respondent_ID, Survey_year) %>% 
  mutate(method_n = n_distinct(paymethcons)) %>% 
  filter(method_n == 1) %>% 
  distinct() %>% 
  group_by(Survey_year, paymethcons) %>% 
  summarise(perc = sum(dweight)*100/total_n) %>% 
  distinct() %>% 
  ungroup() %>% 
  mutate(paymethcons = ifelse(paymethcons %in% c("Cash", "Debit", "Credit", "Other"), paymethcons, "Other")) %>% 
  group_by(Survey_year, paymethcons) %>% 
  summarise(perc = sum(perc))


# Figure 3: Influence on In-person Payment Method Choice #
fig3 <- data19 %>% 
  filter(idtag == 1) %>% 
  select(casenumber, q29, str_subset(names(.), "weight")) %>% 
  mutate(q29 = as.character(q29),
         q29 = gsub("[(0-9)]", "", q29)) %>%
  filter(!str_detect(q29, "Missing")) %>% 
  count(q29, wt = weight) %>% 
  mutate(perc = n*100/sum(n)) %>% 
  select(-n) %>% 
  setNames(c("influence", "2019")) %>% 
  mutate(graph_groups = case_when(
    str_detect(influence, "[Ss]peed") ~ "Transaction speed",
    str_detect(influence, "[Ww]hatever") ~ "Whatever I have with me",
    str_detect(influence, "[Rr]eward") ~ "Reward points",
    str_detect(influence, "own funds") ~ "Prefer own funds",
    str_detect(influence, "[Ss]afety") ~ "Transaction safety/security",
    str_detect(influence, "credit") ~ "Preference for credit",
    str_detect(influence, "charges") ~ "Charges for payment",
    str_detect(influence, "manag") ~ "Budgeting/ \nmanaging finances",
    T ~ "Other"
  )) %>% 
  group_by(graph_groups) %>% 
  summarise(`2019` = sum(`2019`)) %>% 
  left_join(., merged_data %>% 
              filter(idtag_diarycrosswave == 1, inper50payinfl_main != "Missing") %>% 
              select(Respondent_ID, inper50payinfl_main, weight) %>% 
              count(inper50payinfl_main, wt = weight) %>% 
              mutate(perc = n*100/sum(n)) %>% 
              setNames(c("influence", "2022", "perc")) %>% 
              mutate(graph_groups = case_when(
                str_detect(influence, "[Ss]peed") ~ "Transaction speed",
                str_detect(influence, "[Ww]hatever") ~ "Whatever I have with me",
                str_detect(influence, "[Rr]eward") ~ "Reward points",
                str_detect(influence, "own funds") ~ "Prefer own funds",
                str_detect(influence, "[Ss]afety") ~ "Transaction safety/security",
                str_detect(influence, "credit") ~ "Preference for credit",
                str_detect(influence, "[Ss]urcharges") ~ "Charges for payment",
                str_detect(influence, "manag") ~ "Budgeting/ \nmanaging finances",
                T ~ "Other"
              )) %>% 
              group_by(graph_groups) %>% 
              summarise(perc = sum(perc)), by = "graph_groups") %>% 
  arrange(-perc) %>% 
  setNames(c("graph_groups", "2019", "2022")) %>% 
  mutate(order = 1:nrow(.)) %>% 
  pivot_longer(., cols = c("2019", "2022"))


# Figure 4: Merchant Payments Acceptance #
fig4_small_acceptance <- merged_data %>% 
  filter(idtag_diarycrosswave == 1) %>% 
  select(Respondent_ID, dweight, inpersmlpay_1:inpersmlpay_5) %>% 
  pivot_longer(., cols = inpersmlpay_1:inpersmlpay_5) %>% 
  filter(value!="Missing")%>%
  group_by(name) %>% 
  count(value, wt = dweight) %>% 
  mutate(perc_2022 = n*100/sum(n)) %>% 
  mutate(name = case_when(
    name == "inpersmlpay_1" ~ "Cash",
    name == "inpersmlpay_2" ~ "Debit",
    name == "inpersmlpay_3" ~ "Credit",
    name == "inpersmlpay_4" ~ "Cheque",
    name == "inpersmlpay_5" ~ "BNPL",
  )) %>%
  mutate(order = case_when(
    name == "Cash" ~ "2",
    name == "Debit" ~ "1",
    name == "Credit" ~ "3",
    name == "Cheque" ~ "5",
    name == "BNPL" ~ "4",
  )) %>%
  select(-n)

fig4_large_acceptance <- merged_data %>% 
  filter(idtag_diarycrosswave == 1) %>% 
  select(Respondent_ID, dweight, inperlrgpay_1:inperlrgpay_5) %>% 
  pivot_longer(., cols = inperlrgpay_1:inperlrgpay_5) %>% 
  filter(value!="Missing")%>%
  mutate(value=case_when(
    value=="Very Important" ~ "Very important",
    T~value
  ))%>%
  group_by(name) %>% 
  count(value, wt = dweight) %>% 
  mutate(perc_2022 = n*100/sum(n)) %>% 
  mutate(name = case_when(
    name == "inperlrgpay_1" ~ "Cash",
    name == "inperlrgpay_2" ~ "Debit",
    name == "inperlrgpay_3" ~ "Credit",
    name == "inperlrgpay_4" ~ "Cheque",
    name == "inperlrgpay_5" ~ "BNPL",
  )) %>%
  mutate(order = case_when(
    name == "Cash" ~ "3",
    name == "Debit" ~ "1",
    name == "Credit" ~ "2",
    name == "Cheque" ~ "5",
    name == "BNPL" ~ "4",
  )) %>%
  select(-n)

fig4 <- rbind(fig4_small_acceptance, fig4_large_acceptance) %>% 
  ungroup() %>% 
  select(-order)



### Section 4: Cash ###

## Figure 6: Cash Payments ##
# In-person
fig6_inperson <- diary_full %>% 
  filter(paymeth_crosswave != "Missing", !is.na(paymeth_crosswave), paymeth_crosswave != "Transport card", payamt < 9999, payamt >= 0, paychan_crosswave == "In person") %>% 
  filter(!(paypurp_crosswave %in% c("Cash deposit", "Transfers to a family member or friend", "Transfer within own accounts"))) %>% 
  group_by(Survey_year, paymethcons3) %>% 
  mutate(counttrans = 1) %>% 
  summarise(number = sum(dweight*counttrans),
            value = sum(payamt*dweight)) %>% 
  group_by(Survey_year) %>% 
  mutate(number = number*100/sum(number),
         value = value*100/sum(value)) %>% 
  filter(paymethcons3 == "Cash")

# All payments
fig6_allpayments <- diary_full %>% 
  filter(paymeth_crosswave != "Missing", !is.na(paymeth_crosswave),  paymeth_crosswave != "Transport card", payamt < 9999, payamt >= 0) %>% 
  filter(!(paypurp_crosswave %in% c("Cash deposit", "Transfers to a family member or friend", "Transfer within own accounts"))) %>% 
  group_by(Survey_year, paymethcons3) %>% 
  mutate(counttrans = 1) %>% 
  summarise(number = sum(dweight*counttrans),
            value = sum(payamt*dweight)) %>% 
  group_by(Survey_year) %>% 
  mutate(number = number*100/sum(number),
         value = value*100/sum(value)) %>% 
  filter(paymethcons3 == "Cash")


## Figure 7: Cash Payments by Transaction Size ##
fig7 <- diary_full %>% 
  filter(!is.na(paymeth_crosswave), paymeth_crosswave !="Missing", !(paychanx %in% c("Online","Offline")), 
         payamtr >= 1, payamtr <= 100) %>% # Filter out certain payments
  filter(!(paypurp_crosswave %in% c("Cash deposit","Transfers to a family member or friend",
                                    "Transfer within own accounts",
                                    "Transfers to a family member or friend"))) %>%
  select(Survey_year, paymethcons2, payamtr, dweight) %>% 
  mutate(counttrans = 1) %>% 
  rbind(., data.frame(Survey_year = c(rep(2007, times = 200),
                                      rep(2010, times = 200),
                                      rep(2013, times = 200),
                                      rep(2016, times = 200),
                                      rep(2019, times = 200),
                                      rep(2022, times = 200)),
                      paymethcons2 = rep(c(rep("Cash", times = 100),
                                           rep("Card", times = 100)), times = 6), 
                      payamtr = rep(seq(1, 100, by = 1), times = 12),
                      dweight = rep(0, times = 1200), 
                      counttrans = rep(NA, times = 1200))) %>% # Add in a 0 weighted fake transaction for each payamt and paymethcons2
  group_by(Survey_year, paymethcons2, payamtr) %>% 
  summarise(number = sum(dweight*counttrans, na.rm = T)) %>% 
  group_by(Survey_year, payamtr) %>% 
  mutate(perc = number*100/sum(number)) %>% 
  group_by(Survey_year, paymethcons2) %>% 
  mutate(movavg = smooth::cma(perc, order = 5)$fitted) %>% 
  ungroup() %>% 
  filter(Survey_year %in% c(2007, 2019, 2022), paymethcons2 == "Cash") %>% 
  select(Survey_year, payamtr, movavg)


## Figure 8: In-person Cash Use (across demographics) ##
# by age
fig8.age <- diary_full %>% 
  filter(paymeth_crosswave >= 0, payamt >= 0, payamt < 9999, paymethcons3 != "Missing", paychan_crosswave == "In person") %>% # Filter out certain payments, and filter on cash only
  filter(!(paypurp_crosswave %in% c("Cash deposit","Transfers to a family member or friend","Transfer within own accounts"))) %>% 
  group_by(Respondent_ID, agebin, Survey_year, dweight, paymethcons3) %>%
  count() %>% # count number in each category
  ungroup() %>%
  mutate(weighted_num = n*dweight) %>%  # calculate percentage contribution of each individual
  group_by(agebin, paymethcons3, Survey_year) %>% # group back by Survey_year and payment method
  summarise(total = sum(weighted_num)) %>%
  ungroup() %>% group_by(Survey_year, agebin) %>%
  mutate(total = total*100/sum(total)) %>%
  filter(paymethcons3 == "Cash") %>% 
  rename(., var = agebin) %>% 
  ungroup() %>% 
  mutate(panel = "By age")

# by age-adjusted income quartile
fig8.agehhincq <- diary_full %>% 
  filter(paymeth_crosswave >= 0, payamt >= 0, payamt < 9999, paymethcons3 != "Missing", paychan_crosswave == "In person") %>% # Filter out certain payments, and filter on cash only
  filter(!(paypurp_crosswave %in% c("Cash deposit","Transfers to a family member or friend","Transfer within own accounts"))) %>% 
  group_by(Respondent_ID, agehhincq, Survey_year, dweight, paymethcons3) %>%
  count() %>% # count number in each category
  ungroup() %>%
  mutate(weighted_num = n*dweight) %>%  # calculate percentage contribution of each individual
  group_by(agehhincq, paymethcons3, Survey_year) %>% # group back by Survey_year and payment method
  summarise(total = sum(weighted_num)) %>%
  ungroup() %>% group_by(Survey_year, agehhincq) %>%
  mutate(total = total*100/sum(total)) %>%
  filter(paymethcons3 == "Cash") %>%
  rename(., var = agehhincq) %>% 
  na.omit() %>% 
  ungroup() %>% 
  mutate(panel = "By income")


# by location
fig8.location <- diary_full %>% 
  filter(paymeth_crosswave >= 0, payamt >= 0, payamt < 9999, paymethcons3 != "Missing", paychan_crosswave == "In person") %>% # Filter out certain payments, and filter on cash only
  filter(!(paypurp_crosswave %in% c("Cash deposit","Transfers to a family member or friend","Transfer within own accounts"))) %>% 
  group_by(Respondent_ID, region, Survey_year, dweight, paymethcons3) %>%
  count() %>% # count number in each category
  ungroup() %>%
  mutate(weighted_num = n*dweight) %>%  # calculate percentage contribution of each individual
  group_by(region, paymethcons3, Survey_year) %>% # group back by Survey_year and payment method
  summarise(total = sum(weighted_num)) %>%
  ungroup() %>% 
  group_by(Survey_year, region) %>%
  mutate(total = total*100/sum(total)) %>%
  filter(paymethcons3 == "Cash") %>%
  rename(., var = region) %>% 
  na.omit() %>% 
  ungroup() %>% 
  mutate(panel = "By location")

# full graph data
fig8 <- rbind(fig8.age, fig8.agehhincq, fig8.location)


## Figure 9: Cash Use Intensity ##
fig9 <- diary_full %>% 
  filter(idtag_diarycrosswave == 1, number_cash_user != "Unidentified") %>% 
  mutate(altgroups = case_when(
    number == 0 ~ "No cash",
    number >= 80 & number != 100 ~ "High-but-not-all",
    number <= 20 & number != 0 ~ "Low-but-some",
    number == 100 ~ "All cash",
    T ~ "Intermediate")) %>% 
  group_by(Survey_year, altgroups) %>% 
  summarise(number = sum(weight)) %>% 
  group_by(Survey_year) %>% 
  mutate(percent = number*100/sum(number)) %>% 
  ungroup()

## Figure 10: Effects of No Cash ##

fig10 <- merged_data %>% 
  filter(idtag_diarycrosswave == 1, !is.na(cashnonaccept), number_cash_user != "Unidentified", number_cash_user != "Intermediate cash users") %>% 
  mutate(altgroups = case_when(
    number == 0 ~ "No cash",
    number >= 80 & number != 100 ~ "High-but-not-all",
    number <= 20 & number != 0 ~ "Low-but-some",
    number == 100 ~ "All cash",
    T ~ "Intermediate")) %>% 
  group_by(cashnonaccept, number_cash_user) %>% 
  count(wt = dweight) %>% 
  group_by(number_cash_user) %>% 
  mutate(perc = n*100/sum(n)) %>% 
  mutate(graph_groups = case_when(
    str_detect(cashnonaccept, "No") | str_detect(cashnonaccept, "minor inconvenience") ~ "No problem/minor inconvenience",
    str_detect(cashnonaccept, "major inconvenience") | str_detect(cashnonaccept, "genuine hardship") ~ "Major inconvenience/genuine hardship",
    T ~ "Don't know"
  )) %>% 
  group_by(graph_groups, number_cash_user) %>% 
  summarise(perc = sum(perc)) %>% 
  ungroup()


## Figure 11: Why Do You Need To Use Cash? ##
fig11 <- merged_data %>% 
  filter(idtag_diarycrosswave == 1, !is.na(cashnonaccept), cashusereasonmain != "Missing", !is.na(cashusereasonmain),
         str_detect(cashnonaccept, "major inconvenience") | str_detect(cashnonaccept, "genuine hardship")) %>% 
  group_by(number_cash_user) %>% 
  count(cashusereasonmain, wt = dweight) %>% 
  mutate(graph_groups = case_when(
    str_detect(cashusereasonmain, "Privacy") | str_detect(cashusereasonmain, "Security") ~ "Security/privacy/trust",
    str_detect(cashusereasonmain, "only accept cash") ~ "Payment method\naccess/use limitation*",
    str_detect(cashusereasonmain, "Budgeting") ~ "Family/budget reasons",
    str_detect(cashusereasonmain, "family and friends") ~ "Family/budget reasons", 
    str_detect(cashusereasonmain, "#Q3_7._14") ~ "Other",
    str_detect(cashusereasonmain, "non-cash way to pay") ~ "Payment method\naccess/use limitation*",
    str_detect(cashusereasonmain, "internet access") ~ "Payment method\naccess/use limitation*",
    str_detect(cashusereasonmain, "get paid in") ~ "Family/budget reasons",
    str_detect(cashusereasonmain, "confident") ~ "Payment method\naccess/use limitation*",
    str_detect(cashusereasonmain, "don't trust") ~ "Security/privacy/trust",
    str_detect(cashusereasonmain, "disability") ~ "Payment method\naccess/use limitation*", 
    str_detect(cashusereasonmain, "_14") ~ "Other")) %>% 
  group_by(graph_groups, number_cash_user) %>% 
  summarise(n = sum(n)) %>%
  group_by(number_cash_user) %>%
  mutate(perc = n*100/sum(n)) %>% 
  mutate(order = case_when(
    graph_groups == "Payment method\naccess/use limitation" ~ 1,
    graph_groups == "Security/privacy/trust" ~ 2,
    graph_groups == "Family/budget reasons" ~ 3
  )) %>% 
  ungroup()


## Figure 12: Has Internet-compatible Mobile Phone ## 
fig12_2022 <- merged_data %>% 
  filter(idtag_diarycrosswave == 1, onlinemob != "Missing", !is.na(onlinemob)) %>% 
  select(Respondent_ID, number_cash_user, onlinemob, dweight) %>% 
  group_by(number_cash_user, onlinemob) %>% 
  summarise(perc = sum(dweight)*100) %>% 
  group_by(number_cash_user) %>% 
  mutate(perc = perc*100/sum(perc)) %>% 
  pivot_wider(., names_from = number_cash_user, values_from = perc) %>% 
  left_join(., merged_data %>% 
              filter(idtag_diarycrosswave == 1, onlinemob != "Missing", !is.na(onlinemob)) %>% 
              select(Respondent_ID, onlinemob, dweight) %>% 
              group_by( onlinemob) %>% 
              summarise(perc = sum(dweight)*100) %>% 
              mutate(`All respondents` = perc*100/sum(perc)) %>% 
              select(-perc), by = "onlinemob") %>% 
  pivot_longer(., cols = names(.)[-1]) %>% 
  filter(name != "Unidentified", onlinemob == "Yes") %>%
  select(-onlinemob) %>%
  mutate(year = "2022")

# Create 2019 data with low/medium/high cash user definition
fig12_2019 <- data19 %>% 
  filter(idtag == 1, mobint != "NA (-1)", mobint != "Missing (-5)") %>%
  select(casenumber, weight, mobint) %>%
  mutate(casenumber = as.character(casenumber + 5000000)) %>%
  left_join(., diary_full %>% 
              filter(idtag_diarycrosswave == 1, Survey_year == 2019) %>%
              select(Respondent_ID, number_cash_user) %>%
              rename(., "casenumber" = "Respondent_ID") %>%
              mutate(casenumber = as.character(casenumber)), by = "casenumber") %>%
  group_by(number_cash_user) %>%
  count(mobint, wt = weight) %>%
  mutate(perc = n*100/sum(n)) %>%
  filter(str_detect(mobint, "Y")) %>%
  select(-mobint, -n) %>%
  filter(number_cash_user != "Unidentified") %>%
  rbind(., 
        data19 %>% 
          filter(idtag == 1) %>%
          select(casenumber, weight, mobint) %>%
          count(mobint, wt=weight) %>%
          mutate(perc = n*100/sum(n)) %>%
          filter(str_detect(mobint, "Y")) %>%
          select(perc) %>%
          mutate(number_cash_user = "All respondents")
  ) %>%
  mutate(year = "2019") %>%
  setNames(c("name", "value", "year"))

# Join the 2019 and 2022 data
fig12 <- rbind(graph_2022, graph_2019)


## Figure 13: Wallet Cash Holdings ##
bn13 <- data13 %>% 
  select(id, pilot, paste0("endbn", c(5, 10, 20, 50, 100)), str_subset(names(.), "weight")) %>%
  mutate(endbn5 = as.numeric(as.character(endbn5)),
         endbn10 = as.numeric(as.character(endbn10)),
         endbn20 = as.numeric(as.character(endbn20)),
         endbn50 = as.numeric(as.character(endbn50)),
         endbn100 = as.numeric(as.character(endbn100))) %>%
  mutate(endHold = endbn5*5 + endbn10*10 + endbn20*20 + endbn50*50 + endbn100*100) %>%
  select(-pilot, -weight1, -weight3) %>%
  rename(Denom5 = endbn5,
         Denom10 = endbn10,
         Denom20 = endbn20,
         Denom50 = endbn50,
         Denom100 = endbn100,
         dweight = nweight) %>% 
  distinct() %>% 
  mutate(year = 2013)

bn16 <- data16 %>% 
  select(id, str_subset(names(.), "Denom"), str_subset(names(.), "weight")) %>%
  mutate(id = id + 4000000) %>% 
  left_join(., dataX %>% select(id, dweight), by = "id") %>%
  mutate(Denom5 = as.numeric(as.character(Denom5)),
         Denom10 = as.numeric(as.character(Denom10)),
         Denom20 = as.numeric(as.character(Denom20)),
         Denom50 = as.numeric(as.character(Denom50)),
         Denom100 = as.numeric(as.character(Denom100))) %>% 
  mutate(endHold = Denom5*5 + Denom10*10 + Denom20*20 + Denom50*50 + Denom100*100) %>%
  distinct() %>% 
  mutate(year = 2016) %>%
  select(names(bn13))

bn19 <- data19 %>% 
  select(casenumber, str_subset(names(.), "denom")) %>%
  mutate(casenumber = casenumber + 5000000) %>% 
  rename(id = casenumber) %>%
  left_join(., dataX %>% select(id, dweight), by = "id") %>%
  mutate(Denom5 = as.numeric(as.character(postdenom5)),
         Denom10 = as.numeric(as.character(postdenom10)),
         Denom20 = as.numeric(as.character(postdenom20)),
         Denom50 = as.numeric(as.character(postdenom50)),
         Denom100 = as.numeric(as.character(postdenom100))) %>% 
  mutate(endHold = Denom5*5 + Denom10*10 + Denom20*20 + Denom50*50 + Denom100*100) %>%
  distinct() %>% 
  mutate(year = 2019) %>% 
  select(names(bn13))

bn22 <- merged_data %>% 
  filter(idtag_diarycrosswave == 1) %>% 
  select(Respondent_ID, starts_with("postdonom"), dweight, banknotes, Survey_year) %>% 
  setNames(names(bn13))

### Join together to create long dataset
bn <- rbind(bn13, bn16, bn19, bn22) 

# Create graph data
fig13 <- bn %>% 
  na.omit() %>% 
  filter(endHold != -5, !is.na(endHold), endHold != "Missing") %>% 
  mutate(bin = case_when(
    as.numeric(endHold) == 0 ~ "$0",
    as.numeric(endHold) > 0 & as.numeric(endHold) <= 20 ~ "$1-20",
    as.numeric(endHold) > 20 & as.numeric(endHold) <= 50 ~ "$21-50",
    as.numeric(endHold) > 50 & as.numeric(endHold) <= 100 ~ "$51-100",
    as.numeric(endHold) > 100 ~ "$101+"
  )) %>% 
  select(id, year, dweight, endHold, bin) %>% 
  group_by(year) %>% 
  count(bin, wt = dweight) %>% 
  mutate(perc = n*100/sum(n)) %>% 
  mutate(order = case_when(
    bin == "$0" ~ 1,
    bin == "$1-20" ~ 2,
    bin == "$21-50" ~ 3,
    bin == "$51-100" ~ 4,
    bin == "$101+" ~ 5
  )) %>% 
  ungroup()



## Figure 14: Most Preferred Access Points ##
fig14 <- merged_data %>% 
  filter(idtag_diarycrosswave == 1, !is.na(cashaccessmain), cashaccessmain != "Missing", number_cash_user != "Unidentified") %>% 
  select(Respondent_ID, dweight, number_cash_user, cashaccessmain) %>% 
  group_by(number_cash_user) %>% 
  count(cashaccessmain, wt = dweight) %>% 
  mutate(perc = n*100/sum(n)) %>%
  mutate(graph_groups = ifelse(cashaccessmain %in% c("ATM", "Bank branch over the counter", "Cash out from a store", "From a friend or family member", "Post office (i.e. Bank@Post)"),
                               cashaccessmain, "Other")) %>% 
  group_by(graph_groups, number_cash_user) %>% 
  summarise(perc = sum(perc)) %>% 
  mutate(graph_groups = case_when(
    str_detect(graph_groups, "Bank branch") ~ "Bank branch",
    str_detect(graph_groups, "Cash out") ~ "Cash out",
    str_detect(graph_groups, "family member") ~ "Friends/family", 
    T ~ graph_groups)) %>% 
  pivot_wider(., id_cols = graph_groups, names_from = number_cash_user, values_from = perc) %>% 
  left_join(., merged_data %>% 
              filter(idtag_diarycrosswave == 1, !is.na(cashaccessmain), cashaccessmain != "Missing") %>% 
              select(Respondent_ID, dweight, cashaccessmain) %>% 
              count(cashaccessmain, wt = dweight) %>% 
              mutate(perc = n*100/sum(n)) %>%
              mutate(graph_groups = ifelse(cashaccessmain %in% c("ATM", "Bank branch over the counter", "Cash out from a store", "From a friend or family member", "Post office (i.e. Bank@Post)"),
                                           cashaccessmain, "Other")) %>% 
              group_by(graph_groups) %>% 
              summarise(perc = sum(perc)) %>% 
              mutate(graph_groups = case_when(
                str_detect(graph_groups, "Bank branch") ~ "Bank branch",
                str_detect(graph_groups, "Cash out") ~ "Cash out",
                str_detect(graph_groups, "family member") ~ "Friends/family", 
                T ~ graph_groups)), by = "graph_groups") %>% 
  select(-`Intermediate cash users`) %>% 
  setNames(c("graph_groups", "High cash users", "Low cash users", "perc")) %>% 
  pivot_longer(., cols = names(.)[-1])


# Create ordering variable so it's ascending on the "all respondents" category
fig14 <- mostpreferred_data %>% 
  mutate(order = match(graph_groups, 
                       mostpreferred_data %>% 
                         filter(name == "perc") %>% 
                         arrange(-value) %>% 
                         mutate(graph_groups = factor(graph_groups, levels = .$graph_groups, ordered = T)) %>% 
                         select(graph_groups) %>% 
                         unlist() %>% 
                         levels(.)
  ))

# Rename all respondents data accordingly
fig14 <- mostpreferred_data %>% 
  mutate(name = ifelse(name == "perc", "All respondents", name))


## Figure 15: Access to Cash Services ##
fig15 <- merged_data %>% 
  filter(idtag_diarycrosswave == 1, !is.na(cashaccess), cashaccess != "Missing", !is.na(cashdeposit), cashdeposit != "Missing") %>% 
  select(Respondent_ID, dweight, number_cash_user, cashaccess, cashdeposit) %>% 
  pivot_longer(., cols = c("cashaccess", "cashdeposit")) %>% 
  group_by(name) %>% 
  count(value, wt = dweight) %>% 
  mutate(perc = n*100/sum(n)) %>% 
  mutate(graph_groups = case_when(
    str_detect(value, "don't use") ~ "Don't use",
    str_detect(value, "[Ii]nconvenient") ~ "Inconvenient",
    T ~ "Convenient"
  )) %>% 
  group_by(graph_groups, name) %>% 
  summarise(perc = sum(perc)) %>% 
  mutate(name = ifelse(str_detect(name, "access"), "1Withdrawal", "2Deposit")) %>% 
  ungroup()


fig15_22 <- merged_data %>% 
  filter(idtag_diarycrosswave == 1, !is.na(cashaccess), cashaccess != "Missing", !is.na(cashdeposit), cashdeposit != "Missing") %>% 
  select(Respondent_ID, dweight, number_cash_user, cashaccess, cashdeposit) %>% 
  pivot_longer(., cols = c("cashaccess", "cashdeposit")) %>% 
  filter(!str_detect(value, "don't use")) %>% 
  group_by(name) %>% 
  count(value, wt = dweight) %>% 
  mutate(perc = n*100/sum(n)) %>% 
  mutate(name = ifelse(str_detect(name, "access"), "withdrawal", "deposit")) %>% 
  ungroup() %>% 
  select(-n) %>% 
  mutate(Survey_year = "2022") %>% 
  mutate(value = as.character(value),
         value = case_when(
           value == "Very convenient" ~ "1Very convenient",
           value == "Convenient" ~ "2Convenient",
           value == "Inconvenient" ~ "3Inconvenient",
           value == "Very inconvenient" ~ "4Very inconvenient"
         ))


fig15_19 <- data19 %>% 
  filter(idtag == 1, q9a != "Missing (-5)", q9b != "Missing (-5)") %>% 
  select(casenumber, weight, q9a, q9b) %>% 
  setNames(c("Respondent_ID", "weight", "withdrawal", "deposit")) %>% 
  pivot_longer(., cols = c("withdrawal", "deposit")) %>% 
  filter(!str_detect(value, "Don't use")) %>% 
  group_by(name) %>% 
  count(value, wt = weight) %>% 
  mutate(perc = n*100/sum(n)) %>% 
  select(-n) %>% 
  ungroup() %>% 
  mutate(value = as.character(value),
         value = case_when(
    parse_number(value) == 1 ~ "1Very convenient",
    parse_number(value) == 2 ~ "2Convenient",
    parse_number(value) == 3 ~ "3Inconvenient",
    parse_number(value) == 4 ~ "4Very inconvenient",
  )) %>% 
  mutate(Survey_year = "2019")

fig15 <- rbind(fig15_19, fig15_22)


## Figure 16: Inconvenience of Cash Access ## 
# redefine remoteness category. Loading 2019 data thru read.dta13 without character strings
data19 %>% 
  filter(idtag == 1) %>% 
  select(casenumber, remoteness) %>% 
  mutate(Respondent_ID = as.numeric(casenumber)+5000000) %>% 
  left_join(., diary_full %>% 
              filter(Survey_year == 2019) %>% 
              select(Respondent_ID, remoteness), by = "Respondent_ID") %>% 
  select(contains("remoteness")) %>% 
  distinct()

data19 <- data19 %>% 
  mutate(remoteness = case_when(
    remoteness == 1 ~ "Major city",
    remoteness == 2 ~ "Inner Regional",
    remoteness == 3 ~ "Outer Regional",
    remoteness == 4 ~ "Remote"
  ))

g.data <- data19 %>% 
  filter(idtag == 1) %>% 
  mutate(major_city = case_when(
    str_detect(remoteness, "Major ") ~ "Major cities", 
    T ~ "Regional")) %>% 
  select(casenumber, weight, major_city, q9a, q9b) %>% 
  filter(q9a != "Missing (-5)", q9b != "Missing (-5)", q9a != "NA (-1)", q9b != "NA (-1)", q9a != "Don't use (0)", q9b != "Don't use (0)") %>% 
  pivot_longer(., cols = c("q9a", "q9b")) %>% 
  group_by(major_city, name) %>%
  count(value, wt = weight) %>% 
  mutate(perc = n*100/sum(n)) %>% 
  mutate(name = ifelse(name == "q9a", "Withdrawal", "Deposit")) %>% 
  filter(str_detect(value, "Inconvenient")) %>% 
  mutate(value = "Inconvenient") %>% 
  group_by(major_city, name) %>% 
  summarise(perc = sum(perc)) %>% 
  mutate(year = 2019) %>% 
  rbind(., merged_data %>% # Join 2019 with 2022
          filter(idtag_diarycrosswave == 1, !is.na(cashaccess), cashaccess != "Missing", !is.na(cashdeposit), 
                 cashdeposit != "Missing", !str_detect(cashaccess, "don't"), !str_detect(cashdeposit, "don't")) %>% 
          mutate(major_city = case_when(
            str_detect(remoteness, "Major ") ~ "Major cities", 
            T ~ "Regional")) %>% 
          select(Respondent_ID, dweight, major_city, cashaccess, cashdeposit) %>% 
          pivot_longer(., cols = c("cashaccess", "cashdeposit")) %>% 
          group_by(name, major_city) %>% 
          count(value, wt = dweight) %>% 
          mutate(perc = n*100/sum(n)) %>% 
          mutate(graph_groups = case_when(
            str_detect(value, "don't use") ~ "Don't use",
            str_detect(value, "[Ii]nconvenient") ~ "Inconvenient",
            T ~ "Convenient"
          )) %>% 
          group_by(graph_groups, name, major_city) %>% 
          summarise(perc = sum(perc)) %>% 
          mutate(name = ifelse(name == "cashaccess", "Withdrawal", "Deposit")) %>% 
          filter(graph_groups == "Inconvenient") %>% 
          mutate(year = 2022) %>% 
          ungroup() %>% 
          select(-graph_groups))

## Grab the high/low cash user stats
g.data.19 <- data19 %>% 
  filter(idtag == 1) %>% 
  mutate(major_city = case_when(
    str_detect(remoteness, "Major ") ~ "Major cities", 
    T ~ "Regional")) %>% 
  select(casenumber, weight, major_city, q9a, q9b) %>% 
  filter(q9a != "Missing (-5)", q9b != "Missing (-5)", q9a != "NA (-1)", q9b != "NA (-1)", q9a != "Don't use (0)", q9b != "Don't use (0)") %>% 
  # join the cash user group memberships on 
  rename(., Respondent_ID = casenumber) %>% 
  mutate(Respondent_ID = as.numeric(Respondent_ID) + 5000000) %>% 
  left_join(., diary_full %>% 
              filter(idtag_diarycrosswave == 1) %>% 
              select(Respondent_ID, number_cash_user), by = "Respondent_ID") %>% 
  pivot_longer(., cols = c("q9a", "q9b")) %>%
  mutate(value = ifelse(parse_number(as.character(value)) %in% c(1, 2), "Convenient", "Inconvenient")) %>% 
  group_by(name, number_cash_user) %>% 
  count(value, wt = weight) %>% 
  mutate(percent = n*100/sum(n)) %>% 
  mutate(name = ifelse(name == "q9a", "Withdrawal", "Deposit")) %>% 
  filter(number_cash_user %in% c("Low cash users", "High cash users"), value == "Inconvenient") %>% 
  mutate(year = 2019)

g.data.22 <- merged_data %>% # Join 2019 with 2022
  filter(idtag_diarycrosswave == 1, !is.na(cashaccess), cashaccess != "Missing", !is.na(cashdeposit), 
         cashdeposit != "Missing", !str_detect(cashaccess, "don't"), !str_detect(cashdeposit, "don't")) %>% 
  select(Respondent_ID, dweight, number_cash_user, cashaccess, cashdeposit) %>% 
  pivot_longer(., cols = c("cashaccess", "cashdeposit")) %>% 
  mutate(value = ifelse(value %in% c("Very convenient", "Convenient"), "Convenient", "Inconvenient")) %>% 
  group_by(name, number_cash_user) %>% 
  count(value, wt = dweight) %>% 
  mutate(percent = n*100/sum(n)) %>% 
  mutate(name = ifelse(name == "cashaccess", "Withdrawal", "Deposit")) %>% 
  filter(value == "Inconvenient", number_cash_user %in% c("Low cash users", "High cash users")) %>% 
  mutate(year = 2022)

g.data.19.all <- data19 %>% 
  filter(idtag == 1) %>% 
  select(casenumber, weight, q9a, q9b) %>% 
  filter(q9a != "Missing (-5)", q9b != "Missing (-5)", q9a != "NA (-1)", q9b != "NA (-1)", q9a != "Don't use (0)", q9b != "Don't use (0)") %>% 
  # join the cash user group memberships on 
  rename(., Respondent_ID = casenumber) %>% 
  mutate(Respondent_ID = as.numeric(Respondent_ID) + 5000000) %>% 
  left_join(., diary_full %>% 
              filter(idtag_diarycrosswave == 1) %>% 
              select(Respondent_ID, number), by = "Respondent_ID") %>% 
  mutate(number_cash_user = case_when(
    number == 100 ~ "All cash",
    number == 0 ~ "No cash"
  )) %>% 
  pivot_longer(., cols = c("q9a", "q9b")) %>%
  mutate(value = ifelse(parse_number(as.character(value)) %in% c(1, 2), "Convenient", "Inconvenient")) %>% 
  group_by(name, number_cash_user) %>% 
  count(value, wt = weight) %>% 
  mutate(percent = n*100/sum(n)) %>% 
  mutate(name = ifelse(name == "q9a", "Withdrawal", "Deposit")) %>% 
  filter(number_cash_user %in% c("All cash", "No cash"), value == "Inconvenient") %>% 
  mutate(year = 2019)

g.data.22.all <- merged_data %>% # Join 2019 with 2022
  filter(idtag_diarycrosswave == 1, !is.na(cashaccess), cashaccess != "Missing", !is.na(cashdeposit), 
         cashdeposit != "Missing", !str_detect(cashaccess, "don't"), !str_detect(cashdeposit, "don't")) %>% 
  mutate(number_cash_user = case_when(
    number == 100 ~ "All cash",
    number == 0 ~ "No cash"
  )) %>% 
  select(Respondent_ID, dweight, number_cash_user, cashaccess, cashdeposit) %>% 
  pivot_longer(., cols = c("cashaccess", "cashdeposit")) %>% 
  mutate(value = ifelse(value %in% c("Very convenient", "Convenient"), "Convenient", "Inconvenient")) %>% 
  group_by(name, number_cash_user) %>% 
  count(value, wt = dweight) %>% 
  mutate(percent = n*100/sum(n)) %>% 
  mutate(name = ifelse(name == "cashaccess", "Withdrawal", "Deposit")) %>% 
  filter(value == "Inconvenient", number_cash_user %in% c("No cash", "All cash")) %>% 
  mutate(year = 2022)

graph.data <- rbind(g.data.19, g.data.22) %>% 
  rename(., group = number_cash_user) %>% 
  select(-value, -n) %>% 
  rbind(., g.data %>% rename(., group = major_city, percent = perc)) %>% 
  rbind(., rbind(g.data.19.all, g.data.22.all) %>% 
          rename(., group = number_cash_user) %>% 
          select(-value, -n)) %>% 
  mutate(group = factor(group, levels = c("All cash", "No cash", "High cash users", "Low cash users", "Regional", "Major cities")))


### Section 5: Card Use ###

## Figure 17: Card Payments (Share of number of payments in each category) ##
fig17 <- diary_full %>% 
  filter(payamt < 9999, payamt >= 1, !is.na(paymeth_crosswave), paymeth_crosswave != "Missing", paymethcons3 != "Transport card") %>% 
  filter(!(paypurp_crosswave %in% c("Cash deposit", "Transfers to a family member or friend", "Transfer within own accounts"))) %>% 
  group_by(agebin, Survey_year, paymethcons2) %>% 
  summarise(number = sum(weight)) %>% 
  group_by(Survey_year, agebin) %>% 
  mutate(perc = number*100/sum(number)) %>% 
  filter(paymethcons2 == "Card") %>% 
  rename(., "var" = "agebin") %>% 
  mutate(panel = 1) %>% 
  filter(Survey_year %in% c(2007, 2019, 2022)) %>% 
  rbind(., diary_full %>% 
          filter(payamt < 9999, payamt >= 1, !is.na(paymeth_crosswave), paymeth_crosswave != "Missing", paymethcons3 != "Transport card") %>% 
          filter(!(paypurp_crosswave %in% c("Cash deposit", "Transfers to a family member or friend", "Transfer within own accounts"))) %>% 
          group_by(agehhincq, Survey_year, paymethcons2) %>% 
          summarise(number = sum(weight)) %>% 
          group_by(Survey_year, agehhincq) %>% 
          mutate(perc = number*100/sum(number)) %>% 
          filter(paymethcons2 == "Card") %>% 
          rename(., "var" = "agehhincq") %>% 
          mutate(panel = 2) %>% 
          filter(Survey_year %in% c(2007, 2019, 2022), !is.na(var)))


## Figure 18: Debit and Credit Card Payments ##
fig18_left <- diary_full %>% 
  filter(!is.na(paymeth_crosswave), paymeth_crosswave != "NA", payamt > 0, payamt < 9999, paymeth_crosswave != "Missing") %>% # Filter out certain payments
  filter(!(paypurp_crosswave %in% c("Cash deposit","Transfers to a family member or friend","Transfer within own accounts"))) %>% 
  filter(ccown != "Missing") %>% 
  mutate(counttrans = 1) %>% 
  group_by(Survey_year, agehhincq, paymethcons3) %>% 
  summarise(number = sum(counttrans*weight)) %>% 
  group_by(Survey_year, agehhincq) %>% 
  mutate(perc = number*100/sum(number)) %>% 
  filter(Survey_year == 2022, paymethcons3 %in% c("Credit", "Debit")) %>% 
  ungroup() %>% 
  select(agehhincq, paymethcons3, perc) %>% 
  mutate(facet = "agehhincq") %>% 
  rename(., panel = agehhincq)

fig18_right <- diary_full %>% 
  filter(!is.na(paymeth_crosswave), paymeth_crosswave != "NA", payamt > 0, payamt < 9999, paymeth_crosswave != "Missing") %>% # Filter out certain payments
  filter(!(paypurp_crosswave %in% c("Cash deposit","Transfers to a family member or friend","Transfer within own accounts"))) %>% 
  filter(ccown != "Missing") %>% 
  mutate(counttrans = 1) %>% 
  group_by(Survey_year, agebin, paymethcons3) %>% 
  summarise(number = sum(counttrans*weight)) %>% 
  group_by(Survey_year, agebin) %>% 
  mutate(perc = number*100/sum(number)) %>% 
  filter(Survey_year == 2022, paymethcons3 %in% c("Credit", "Debit")) %>% 
  ungroup() %>% 
  select(agebin, paymethcons3, perc) %>% 
  mutate(facet = "agebin") %>% 
  rename(., panel = agebin)

fig18 <- rbind(fig18_left, fig18_right)

## Figure 19: Change in Debit and Credit Card Payments ##
fig19 <- diary_full %>% 
  filter(!is.na(paymeth_crosswave), payamt >= 0, payamt < 9999, paymethcons3 != "Missing") %>% # Filter out certain payments
  filter(!(paypurp_crosswave %in% c("Cash deposit","Transfers to a family member or friend","Transfer within own accounts"))) %>% 
  group_by(paymethcons3, Survey_year, agebin) %>% # group by years, and payment method
  mutate(counttrans = 1) %>% 
  summarise(number = sum(counttrans*dweight)) %>% 
  mutate(graph_groups = ifelse(paymethcons3 %in% c("Credit", "Debit", "Cash"), paymethcons3, "Other")) %>% 
  group_by(Survey_year, agebin, graph_groups) %>%
  summarise(number = sum(number)) %>% 
  group_by(Survey_year, agebin) %>% 
  mutate(perc = number*100/sum(number)) %>% 
  select(-number) %>% 
  pivot_wider(., names_from = Survey_year, values_from = perc) %>% 
  mutate(diff = `2022` - `2019`) %>% 
  filter(graph_groups %in% c("Credit", "Debit"))


## Figure 20: Debit Card Use by Age ##
fig20 <- diary_full %>% 
  filter(!is.na(paymeth_crosswave), paymeth_crosswave != "NA", payamt > 0, payamt < 9999, paymeth_crosswave != "Missing") %>% # Filter out certain payments
  filter(!(paypurp_crosswave %in% c("Cash deposit","Transfers to a family member or friend","Transfer within own accounts"))) %>% 
  filter(ccown != "Missing") %>% 
  mutate(counttrans = 1) %>% 
  group_by(Survey_year, ccown, agebin, paymethcons3) %>% 
  summarise(number = sum(counttrans*weight)) %>% 
  filter(Survey_year %in% c(2019, 2022)) %>% 
  group_by(Survey_year, ccown, agebin) %>% 
  mutate(perc = number*100/sum(number)) %>% 
  group_by(Survey_year, agebin, paymethcons3) %>% 
  mutate(all_num = sum(number)) %>% 
  filter(ccown == "Yes") %>% 
  group_by(Survey_year, agebin) %>% 
  mutate(all_perc = all_num*100/sum(all_num)) %>% 
  filter(paymethcons3 == "Debit") %>% 
  ungroup() %>% 
  select(Survey_year, agebin, perc, all_perc) %>% 
  setNames(c("Survey_year", "agebin", "Credit card holders", "All respondents"))


## Figure 21: Card Payments by Channel ##
fig21 <- diary_full %>% 
  filter(paymeth_crosswave != "Missing", paymeth_crosswave != "Transport card", !is.na(paymeth_crosswave), payamt < 9999, payamt >= 0, paymethcons2 == "Card") %>% # Select only card payments
  filter(!(paypurp_crosswave %in% c("Cash deposit", "Transfers to a family member or friend", "Transfer within own accounts"))) %>% 
  filter(paychanx != "Missing") %>% 
  filter(paychan_crosswave == "In person") %>% 
  mutate(counttrans = 1) %>%
  group_by(Survey_year, paychanx) %>% 
  summarise(volume = sum(dweight*counttrans)) %>% 
  group_by(Survey_year) %>% 
  setNames(c("Survey_year", "group", "volume")) %>%  
  filter(group %in% c( "Tap device", "Tap card", "Insert card")) %>%
  mutate(volume = volume*100/sum(volume)) %>%
  ungroup() %>%
  pivot_wider(., names_from = group, values_from = volume)


## Figure 22: In-person payments (Change in share of number of payments by value 2019 to 2022) ##
fig22 <- diary_full  %>% 
  filter(!is.na(paymeth_crosswave), paymeth_crosswave != "NA", payamt > 0, payamt < 9999, paymeth_crosswave != "Missing") %>% # Filter out certain payments
  filter(!(paypurp_crosswave %in% c("Cash deposit","Transfers to a family member or friend","Transfer within own accounts"))) %>% 
  filter(paychanx %in% c("Tap card", "Tap device", "Insert card", "Other in person")) %>% # Filter on in person payments channels
  select(Survey_year, payamtbinr2, starts_with("paychan"), dweight) %>% 
  group_by(Survey_year) %>%  
  group_by(Survey_year, payamtbinr2, paychanx) %>% 
  mutate(counttrans = 1) %>% 
  summarise(perc = round(sum(dweight*counttrans),4)) %>% 
  filter(Survey_year > 2010) %>% 
  group_by(Survey_year, payamtbinr2) %>% 
  mutate(perc = perc*100/sum(perc)) %>% 
  mutate(payamtbinr2 = ifelse(payamtbinr2 == "$1-$5", "$1-$10", payamtbinr2)) %>% 
  pivot_wider(., id_cols = c("paychanx", "payamtbinr2"), names_from = Survey_year, values_from = perc) %>% 
  arrange(parse_number(payamtbinr2)) %>% 
  mutate(delcheck = `2019` - `2016`,
         del      = `2022` - `2019`) %>% 
  ungroup() %>% 
  mutate(order = parse_number(payamtbinr2)) %>% 
  group_by(payamtbinr2) %>% 
  mutate(card = ifelse(paychanx %in% c("Insert card", "Tap card", "Tap device"), "Total card", "No")) %>% 
  group_by(card, payamtbinr2) %>% 
  mutate(`Total card` = ifelse(card == "No", NA, sum(del))) %>% 
  ungroup()

## Figure 23: Mobile Device Use by Age #
fig23_top_panel <- diary_full %>% 
  filter(Survey_year %in% c(2019, 2022), paychanx %in% c("Tap card", "Tap device", "Insert card", "Other in person")) %>% 
  group_by(Respondent_ID, Survey_year, agebin, weight) %>% 
  summarise(tapdev = ifelse("Tap device" %in% paychan2019, 1, 0)) %>% 
  group_by(agebin, Survey_year, tapdev) %>% 
  summarise(perc = sum(weight)) %>% 
  group_by(agebin, Survey_year) %>% 
  mutate(total = perc*100/sum(perc)) %>% 
  filter(tapdev == 1) %>% 
  ungroup() %>% 
  select(Survey_year, agebin, perc)

fig23_bottom_panel <- diary_full %>% 
  filter(Survey_year %in% c(2019, 2022), paychanx %in% c("Tap card", "Tap device", "Insert card", "Other in person")) %>% 
  group_by(Respondent_ID) %>% 
  filter("Tap device" %in% paychan2019) %>% 
  group_by(paychan2019, agebin, Survey_year) %>% 
  count(wt = weight) %>% 
  group_by(agebin, Survey_year) %>% 
  mutate(perc = n*100/sum(n)) %>% 
  filter(paychan2019 == "Tap device") %>% 
  group_by(Survey_year) %>% 
  mutate(order = 1:n()) %>% 
  ungroup() %>% 
  select(Survey_year, agebin, perc)


## Figure 24: Use of Mobile Wallets by Age ##
fig24_2019 <- data19 %>% 
  filter(idtag == 1) %>% 
  select(casenumber, weight, agebin, q26_f, q36) %>% 
  group_by(agebin, q26_f) %>% 
  count(q36, wt = weight) %>% 
  group_by(agebin) %>% 
  mutate(perc = n*100/sum(n)) %>%
  ungroup() %>% 
  mutate(all = n*100/sum(n)) %>% 
  filter(q26_f == "Yes (1)") %>%
  mutate(graph_names = case_when(
    q36 %in% c("generally use last card used (3)", "depending on purchase, choose (4)", " use same card but >1 stored (2)",
               "depending on acceptance, choose (5)", "depending on rewards, choose (6)") ~ "Multiple stored cards",
    q36 == "NA (-1)" ~ "Aware of method, no stored card",
    T ~ "One stored card")) %>% 
  group_by(agebin, graph_names) %>% 
  summarise(perc = sum(perc),
            all = sum(all)) %>% 
  group_by(graph_names) %>% 
  mutate(all = sum(all))%>% 
  pivot_wider(., id_cols = graph_names, names_from = agebin, values_from = c("perc", "all")) %>% 
  pivot_longer(., cols = names(.)[c(2:6, 8)])  %>% 
  select(-c(2:7)) %>% 
  mutate(name = gsub("perc_", "", name)) %>% 
  mutate(name = gsub("_18-29", "", name)) %>% 
  mutate(name = case_when(
    name == "all_1" ~ "All\nrespondents",
    name == "1" ~ "18-29",
    name == "2" ~ "30-39",
    name == "3" ~ "40-49",
    name == "4" ~ "50-64",
    name == "5" ~ "65+")) %>% 
  ungroup() %>% 
  mutate(Survey_year = 2022)


# Do the same for the 2022 data
fig24_2022 <- merged_data %>% 
  filter(idtag_diarycrosswave == 1) %>% 
  select(Respondent_ID, weight, agebin, pmttypeheard_f, starts_with("mobwallet")) %>% 
  pivot_longer(., cols = names(.)[-c(1:4)]) %>% 
  mutate(value = ifelse(is.na(value), "No", value)) %>% 
  group_by(Respondent_ID, value) %>% 
  mutate(n_stored = n()) %>% 
  group_by(Respondent_ID) %>% 
  mutate(graph_names = case_when(
    pmttypeheard_f == "Yes" & value == "Yes" & n_stored > 1 ~ "Multiple stored cards",
    pmttypeheard_f == "Yes" & value == "Yes" & n_stored == 1 ~ "One stored card",
    pmttypeheard_f == "Yes" & !("Yes" %in% value) ~ "Aware of method, no stored card",
    pmttypeheard_f == "No" ~ "Unaware"
  )) %>% 
  select(-name) %>% 
  ungroup() %>% 
  distinct() %>% 
  filter(!is.na(graph_names)) %>% 
  group_by(agebin) %>% 
  count(graph_names, wt = weight) %>% 
  mutate(perc = n*100/sum(n)) %>% 
  ungroup() %>% 
  mutate(all = n*100/sum(n)) %>% 
  group_by(graph_names) %>% 
  mutate(all = sum(all)) %>% 
  filter(graph_names != "Unaware") %>% 
  pivot_wider(., id_cols = graph_names, names_from = agebin, values_from = c("perc", "all")) %>% 
  pivot_longer(., cols = names(.)[c(2:7)]) %>% 
  select(-c(2:5)) %>% 
  mutate(name = gsub("perc_", "", name)) %>% 
  mutate(name = gsub("_18-29", "", name)) %>% 
  mutate(name = ifelse(name == "all", "All\nrespondents", name)) %>% 
  mutate(Survey_year = 2022)

# create top panel graph data

fig24_top <- rbind(fig24_2019, fig24_2022) %>% 
  filter(graph_names != "Aware of method, no stored card")

# Combine and create the change from 2019 to 2022 data
fig24_bottom <- graph_data2022 %>% 
  pivot_wider(., id_cols = graph_names, values_from = value, names_from = name) %>% 
  mutate(year = 2022) %>% 
  rbind(., graph_data2019 %>% 
          pivot_wider(., id_cols = graph_names, values_from = value, names_from = name) %>% 
          mutate(year = 2019)) %>% 
  pivot_longer(., cols = names(.)[2:7]) %>% 
  group_by(graph_names, name) %>% 
  arrange(., .by_group = T) %>% 
  mutate(change = value - dplyr::lead(value, 1)) %>% 
  filter(year == 2022) %>% 
  group_by(name) %>% 
  mutate(`Total change` = sum(change)) %>% 
  select(graph_names, name, `Total change`) %>% 
  filter(graph_names != "Aware of method, no stored card") %>% 
  ungroup()


## Figure 26: Response to Payment Surcharge ##
fig26_2022 <- merged_data %>% 
  filter(idtag_diarycrosswave == 1) %>% 
  select(Respondent_ID, weight, starts_with("surchresp")) %>% 
  pivot_longer(., cols = names(.)[-c(1,2)])  %>% 
  filter(name != "surchresp_other") %>% 
  count(value, wt = weight) %>% 
  mutate(perc = n*100/sum(n)) %>% 
  mutate(value = case_when(
    str_detect(value, "Avoid") ~ "Avoid\nmerchant",
    str_detect(value, "preferred payment method") ~ "Use preferred\nmethod and\npay surcharge",
    str_detect(value, "that is not surcharged") ~ "Choose\nnon-surcharged\nmethod",
    T ~ "Other"
  )) %>% 
  arrange(perc) %>% 
  select(value, perc) %>% 
  setNames(c("Response", "Perc")) %>% 
  mutate(Survey_year = 2022)

fig26_2019 <- data19 %>% 
  filter(idtag == 1) %>% 
  select(casenumber, weight, q25) %>% 
  filter(q25 != "Missing (-5)") %>% 
  mutate(q25 = case_when(
    str_detect(q25, "Stop using") ~ "Avoid\nmerchant",
    str_detect(q25, "Pay with non-surch") ~ "Choose\nnon-surcharged\nmethod",
    str_detect(q25, "Pay surch") ~ "Use preferred\nmethod and\npay surcharge",
    NA ~ "Other"
  )) %>% 
  count(q25, wt = weight) %>% 
  mutate(perc = n*100/sum(n),
         q25 = ifelse(is.na(q25), "Other", q25)) %>% 
  select(q25, perc) %>% 
  setNames(c("Response", "Perc")) %>% 
  mutate(Survey_year = 2019)

# Combine each years' data
rbind(fig26_2019, fig26_2022)


### Section 6: Dual network debit card ###
# Create DNDC Data tables

#Tabulate dual-network debit card questions
DNDC_data <- merged_data %>%
  filter(idtag_diarycrosswave==1) %>%
  dplyr::mutate(short_pre = case_when(
    dndcown_pre == "I don't know / I'm unsure"~"Unsure",
    dndcown_pre == "I don’t know / I’m unsure"~"Unsure",
    dndcown_pre == "No - I don't have an ATM/debit card that can process payments through more than one network" ~ "No",
    dndcown_pre == "No – I don’t have an ATM/debit card that can process payments through more than one network" ~ "No",
    dndcown_pre == "Yes - I have an ATM/debit card that can process payments through more than one network" ~ "Yes",
    dndcown_pre == "Yes – I have an ATM/debit card that can process payments through more than one network" ~ "Yes"
  ))

DNDC_pre <- DNDC_data %>% 
  filter(dcown=="Yes" & !is.na(dndcown_post), !is.na(dndcown_pre)) %>%
  count(dndcown_pre,wt=dweight) %>%
  mutate(freq=n/sum(n))

short_pre<-c("Unsure","No - I don't have a dual network debit card","Yes - I have a dual network debit card")
short_post<-c("No","Yes")

DNDC_post<-  DNDC_data %>% 
  filter(dcown=="Yes"&!is.na(dndcown_post)) %>%
  count(dndcown_post,wt=dweight) %>%
  mutate(freq=n/sum(n))

DNDC_matter<-  DNDC_data %>% 
  filter(dcown=="Yes"&!is.na(dndcmatter)) %>%
  count(dndcmatter,wt=dweight) %>%
  mutate(freq=n/sum(n))

DNDC_factors<-  DNDC_data %>% 
  filter(dcown=="Yes"&!is.na(dndcpreffactor_1)) %>%
  select(contains("dndcpreffactor"),dweight)%>%
  pivot_longer(!dweight,names_to="factors",values_to="y_n")%>%
  count(factors,y_n,wt=dweight)%>%
  group_by(factors)%>%
  mutate(freq=n/sum(n))%>%
  arrange(y_n,factors)

#preferred network
DNDC_pref<-  DNDC_data %>% 
  mutate(dndcpref = case_when(dndcpref=="Eftpos"~"eftpos",T~dndcpref))%>%
  filter(!is.na(dndcpref)) %>%
  count(dndcpref,wt=dweight) %>%
  mutate(freq=n/sum(n))%>%
  arrange(desc(freq))

#preferred network by favoured network
DNDC_factors_by_network<-  DNDC_data %>% 
  mutate(dndcpref = case_when(dndcpref=="Eftpos"~"eftpos",T~dndcpref))%>%
  filter(dcown=="Yes"&!is.na(dndcpreffactor_1)) %>%
  select(contains("dndcpreffactor"),dweight,dndcpref)%>%
  pivot_longer(cols=-c(dweight,dndcpref),names_to="factors",values_to="y_n")%>%
  count(factors,dndcpref,y_n,wt=dweight)%>%
  group_by(factors,dndcpref)%>%
  mutate(freq=100*n/sum(n))%>%
  arrange(y_n,factors)%>%
  select(-n)%>%
  filter(y_n!="No")%>%
  pivot_wider(names_from = dndcpref,values_from = freq)


#mapping how people responded in the pre-diary to the post-diary responses
DNDC_pre_post<-  DNDC_data %>% 
  filter(dcown=="Yes"&!is.na(dndcown_post),!is.na(dndcown_pre)) %>%
  count(short_pre,dndcown_post,wt=dweight) %>%
  mutate(freq=n/sum(n))

DNDC_flows<-DNDC_pre_post %>%
  select(-n)%>%
  pivot_wider(names_from = dndcown_post,values_from = freq)

## Figure 27: Off the top of your head head, ... ###
fig27 <- DNDC_pre %>% 
  select(-n) %>% 
  mutate(perc = freq*100) %>% 
  select(-freq)

## Figure 28: Check your debit card(s), do you have...
fig28 <- DNDC_pre_post %>% 
  mutate(perc = 100*freq) %>% 
  select(-n, -freq) %>% 
  setNames(c("Pre-diary dndc ownership", "Post-diary dndc onwership", "Perc"))

## Figure 29: Does it matter to you which network ...
fig29 <- DNDC_matter %>% 
  select(-n) %>% 
  mutate(perc = freq*100) %>% 
  select(-freq)


### Section 7:  Online and Automated Payments ###

## Figure 30: Online Payments by Sector ##
fig30 <- diary_full %>% 
  filter(Survey_year %in% c(2019, 2022), !is.na(paypurpcons2), paypurpcons2 != "Missing") %>% 
  filter(paymeth_crosswave != "Missing", paymeth_crosswave !="Transport card", !is.na(paymeth_crosswave), payamt < 9999, payamt >= 0) %>% 
  filter(!(paypurp_crosswave %in% c("Cash deposit", "Transfers to a family member or friend", "Transfer within own accounts"))) %>% 
  mutate(counttrans = 1) %>% 
  group_by(Respondent_ID) %>% 
  mutate(online_user = ifelse(paychanx == "Online", "Online payment", "Not online")) %>% 
  group_by(Survey_year, paypurpcons2, online_user) %>% 
  summarise(n = sum(dweight*counttrans)) %>% 
  group_by(Survey_year, paypurpcons2) %>% 
  mutate(perc = n*100/sum(n)) %>% 
  filter(online_user == "Online payment") %>% 
  ungroup() %>% 
  pivot_wider(., names_from = Survey_year, values_from = perc, id_cols = paypurpcons2) %>% 
  arrange(`2022`) %>% 
  pivot_longer(., cols = names(.)[c(2,3)]) %>% 
  setNames(c("Sector", "Survey_year", "perc"))


## Figure 31: Online Payments by Age ##
# Data
fig31_top <- diary_full %>% 
  filter(Survey_year %in% c(2019, 2022)) %>% 
  group_by(Respondent_ID) %>% 
  mutate(online_user = ifelse("Online" %in% paychanx, "Online user", "Not online user")) %>% 
  select(Respondent_ID, Survey_year, agebin, weight, online_user) %>% 
  distinct() %>% 
  group_by(Survey_year, agebin) %>% 
  count(online_user, wt = weight) %>% 
  mutate(perc = n*100/sum(n)) %>% 
  filter(online_user == "Online user") %>% 
  select(Survey_year, agebin, perc) %>% 
  ungroup()

fig31_bottom <- diary_full %>% 
  filter(Survey_year %in% c(2019, 2022), !is.na(paychan_crosswave), !(is.na(paychan2019))) %>% 
  mutate(counttrans = 1) %>% 
  group_by(Respondent_ID) %>% 
  mutate(online_user = ifelse("Online" %in% paychanx, "Online user", "Not online user")) %>% 
  filter(online_user == "Online user") %>% 
  group_by(Survey_year, agebin, paychan2019, online_user) %>% 
  summarise(n = sum(dweight*counttrans)) %>% 
  group_by(Survey_year, agebin) %>% 
  mutate(perc = n*100/sum(n)) %>% 
  filter(paychan2019 %in% c("Online", "App")) %>% 
  ungroup() %>% 
  select(Survey_year, agebin, paychan2019, perc)


## Figure 32: Automatic Payments ##

# Create total spending
n_respondents <- diary_data %>% 
  filter(idtag_diarycrosswave == 1) %>% 
  dplyr::group_by(Survey_year) %>% 
  summarise(n_respondent = n()) %>% 
  rbind(., merged_data %>% 
          filter(idtag_diarycrosswave == 1) %>% 
          group_by(Survey_year) %>% 
          summarise(n_respondent = nrow(.)))

# Diary payments
fig32_diarypayments <- diary_full %>% 
  left_join(., n_respondents, by = "Survey_year") %>% 
  mutate(paychan = case_when(
    Survey_year== 2019 & paychanx %in% c("Tap card", "Tap device", "Insert Card", "Other in person") ~ "In person",
    Survey_year== 2019 & paymeth_crosswave == "Cash" ~ "In person",
    
    Survey_year== 2019 & paychan2019 == "Online (6)" ~ "Internet (PC/tablet)",
    Survey_year== 2019 & paychan2019 == "App (5)" ~ "Smartphone/SMS",
    Survey_year== 2019 & paychan2019 == "Offline (7)" ~ "Mail",
    T ~ paychanx
  )) %>%
  filter(paychan != "Missing", paychan != "NA", !is.na(paychan), as.numeric(payamt) < 9999, as.numeric(payamt) >= 0) %>% 
  select(-paypurp_crosswave) %>% 
  mutate(counttrans = 1) %>% 
  group_by(paychan, Survey_year) %>% 
  summarise(volume = sum(counttrans*weight, na.rm = T),
            value = sum(as.numeric(payamt)*as.numeric(weight), na.rm = T)) %>% 
  arrange(paychan, Survey_year) %>% 
  pivot_wider(., id_cols = paychan, names_from = Survey_year, values_from = c("value", "volume"))


# Automatic payments
fig32_autopayments <- auto_full %>%
  filter(!(autopurp %in% c("NA", NA, "Missing", "missing"))) %>% 
  mutate(counttrans = 1,
         autopurp = str_to_lower(autopurp)) %>% 
  filter(inweek %in% c(1, "Yes")) %>% 
  group_by(Survey_year, autopurp) %>% 
  summarise(volume = sum(as.numeric(counttrans)*as.numeric(weight), na.rm = T),
            value = sum(as.numeric(autoamt)*as.numeric(weight), na.rm = T)) %>% 
  pivot_wider(., id_cols = autopurp, names_from = Survey_year, values_from = c("volume", "value")) %>% 
  na.omit() %>% 
  rename(., paychan = autopurp)


fig32 <- rbind(fig34_diarypayments, fig34_autopayments) %>% 
  pivot_longer(., cols = names(.)[-1]) %>% 
  mutate(val_vol = ifelse(str_detect(name, "value"), "value", "volume")) %>% 
  mutate(Survey_year = parse_number(name)) %>% 
  group_by(Survey_year, val_vol) %>% 
  mutate(perc = value*100/sum(value, na.rm = T)) %>%
  filter(paychan %in% c("bill", "debt repayment", "other")) %>% 
  select(paychan, Survey_year, val_vol, perc) %>% 
  ungroup()


## Figure 33: Cheque Book Ownership by Age ## 
# Find number of respondents, weighted, in each age category
n_respondents <- merged_data %>% 
  filter(idtag_diarycrosswave == 1) %>% 
  group_by(agebin) %>% 
  summarise(N = sum(weight))

fig33 <- merged_data %>% 
  filter(idtag_diarycrosswave == 1, chqbook != "Missing") %>% 
  select(Respondent_ID, agebin, weight, chqbook) %>% 
  group_by(agebin) %>% 
  count(chqbook, wt = weight) %>% 
  group_by(chqbook, agebin) %>% 
  summarise(n = sum(n)) %>% 
  left_join(., y = n_respondents, by = "agebin") %>% 
  group_by(agebin) %>% # Find percentages across agebins
  mutate(perc = n*100/N) %>%
  filter(chqbook == "Yes") %>% 
  ungroup() %>% 
  select(agebin, perc)


### Section 9: Alternative Payment Methods ###

## Figure 34: Alternative Payment Methods ##
fig34 <- merged_data %>% 
  filter(idtag_diarycrosswave == 1) %>% 
  select(Respondent_ID, 
         dweight, 
         str_subset(names(merged_data), "pmttypeheard"),
         str_subset(names(merged_data), "pmttypeuse")) %>% 
  pivot_longer(., cols = names(.)[-c(1,2)]) %>% 
  group_by(name) %>% 
  count(value, wt = dweight) %>% 
  mutate(perc = n*100/sum(n)) %>% 
  mutate(nice_names = case_when(
    str_detect(name, "_a") ~ "Buy Now, Pay Later",
    str_detect(name, "_b") ~ "PayPal",
    str_detect(name, "_c") ~ "Alipay/WeChat Pay",
    str_detect(name, "_d") ~ "Beem It",
    str_detect(name, "_e") ~ "In-app purchase on mobile phone",
    str_detect(name, "_f") ~ "Mobile device 'tap and go'",
    str_detect(name, "_g") ~ "PayID",
    str_detect(name, "_h") ~ "QR code**",
    str_detect(name, "_i") ~ "Cryptocurrency"
  )) %>% 
  filter(value == "Yes") %>% 
  mutate(use_heard = ifelse(str_detect(name, "heard"), "Heard of method", "Used method*")) %>% 
  pivot_wider(., id_cols = nice_names, names_from = use_heard, values_from = perc) %>% 
  arrange(`Heard of method`) %>% 
  pivot_longer(., cols = names(.)[-c(1,4)]) %>% 
  setNames(c("Alt method", "Heard/Used", "Perc"))


## Figure 35: Reasons for Not Using Alternative Payment Methods ##

# Get heard data
fig35_heard_of <- merged_data %>% 
  filter(idtag_diarycrosswave == 1) %>% 
  select(Respondent_ID, weight, starts_with("pmttypeheard")) %>% 
  pivot_longer(., cols = names(.)[-c(1:2)]) %>% 
  filter(!str_detect(name, "MINA")) %>% 
  setNames(c("Respondent_ID", "weight", "type", "heard")) %>% 
  mutate(type = str_sub(type, -1, -1))

# Get long not used
fig35_not_used <- merged_data %>% 
  filter(idtag_diarycrosswave == 1) %>% 
  select(Respondent_ID, weight, starts_with("notused")) %>% 
  pivot_longer(., cols = names(.)[-c(1:2)]) %>% 
  filter(!str_detect(name, "MINA")) %>% 
  mutate(notusedreason = parse_number(name)) %>% 
  setNames(c("Respondent_ID", "weight", "type", "notused?", "notusedreason")) %>% 
  mutate(type = str_sub(type, -3, -3)) %>% 
  select(-weight)

# Join by type and respondent_id
fig35 <- left_join(heard_of, not_used, by = c("Respondent_ID", "type")) %>% 
  select(Respondent_ID, weight, type, heard, notusedreason, `notused?`) %>% 
  group_by(type, heard, notusedreason) %>% 
  count(`notused?`, wt = weight) %>% 
  mutate(type = case_when(
    type == "a" ~ "BNPL",
    type == "b" ~ "PayPal",
    type == "c" ~ "AliPay/WeChat",
    type == "d" ~ "Beem It",
    type == "e" ~ "In-app",
    type == "f" ~ "Tap and go",
    type == "g" ~ "PayID",
    type == "h" ~ "QR code",
    type == "i" ~ "Cryptocurrency"
  )) %>% 
  mutate(notusedreason = case_when(
    notusedreason == 1 ~ "What I have\nmeets my needs",
    notusedreason %in% c(2, 3) ~ "Don't know how to use/\ndon't have necessary tools",
    notusedreason %in% c(4, 5) ~ "Concerned about privacy\nand/or security",
    notusedreason == 6 ~ "Worried about keeping\ncontrol of spending",
    notusedreason == 7 ~ "Sellers don't\naccept method",
    T ~ "Other"
  )) %>% 
  filter(heard == "Yes", `notused?` == "Yes") %>% 
  group_by(type, notusedreason) %>% 
  summarise(n = sum(n)) %>% 
  group_by(type) %>% 
  mutate(perc = n*100/sum(n))  %>% 
  ungroup() %>% 
  select(type, notusedreason, perc)


## Figure 36: Cryptocurrency Holders ##
fig36 <- merged_data %>% 
  filter(idtag_diarycrosswave == 1) %>% 
  select(Respondent_ID, weight, agebin, gender, starts_with("crypto")) %>%
  group_by(agebin, gender) %>% 
  count(cryptohold, cryptouse, wt=weight) %>% 
  group_by(agebin, gender) %>% 
  mutate(perc = n*100/sum(n))%>%
  filter(cryptohold=="Yes"|cryptouse=="Yes")%>%
  group_by(agebin, gender,cryptouse) %>% 
  summarise(perc=sum(perc)) %>% 
  ungroup()


## Figure 37: BNPL Use by Age ##
fig37_2019 <- data19 %>% 
  filter(idtag == 1) %>% 
  select(casenumber, weight, agebin, q37) %>% 
  group_by(agebin) %>% 
  count(q37, wt = weight) %>% 
  mutate(perc = n*100/sum(n)) %>% 
  group_by(q37) %>% 
  mutate(all = sum(n)) %>% 
  group_by(agebin) %>% 
  mutate(all = all*100/sum(all)) %>% 
  filter(q37 == "Yes (1)") %>% 
  select(-n) %>% 
  pivot_wider(., values_from = c("perc", "all"), names_from = agebin) %>% 
  select(starts_with("perc")) %>%  
  setNames(c("18-29", "30-39", "40-49", "50-64", "65+", "All \nrespondents")) %>% 
  pivot_longer(., cols = names(.)) %>% 
  mutate(year = 2019)

fig37_2022 <- merged_data %>% 
  filter(idtag_diarycrosswave == 1) %>% 
  select(Respondent_ID, weight, agebin, pmttypeuse_a) %>% 
  group_by(agebin) %>% 
  count(pmttypeuse_a, wt = weight) %>% 
  mutate(perc = n*100/sum(n)) %>% 
  group_by(pmttypeuse_a) %>% 
  mutate(all = sum(n)) %>% 
  group_by(agebin) %>% 
  mutate(all = all*100/sum(all)) %>% 
  filter(pmttypeuse_a == 'Yes') %>% 
  select(-n) %>% 
  pivot_wider(., values_from = c("perc", "all"), names_from = agebin)  %>% 
  select(starts_with("perc"), "all_18-29") %>%  
  setNames(c("18-29", "30-39", "40-49", "50-64", "65+", "All \nrespondents")) %>% 
  pivot_longer(., cols = names(.)) %>% 
  mutate(year = 2022)

fig39 <- rbind(fig39_2019, fig39_2022) %>% 
  setNames(c("agebin", "Perc", "Survey_year"))


## Figure 38: Credit Card Ownership by Age and BNPL Usage ##
fig38_bnpl_ccown<- merged_data %>% filter(idtag_diarycrosswave == 1)%>%
  count(pmttypeuse_a,ccown,agebin,wt=weight) %>% 
  group_by(agebin,pmttypeuse_a) %>%
  mutate(freq=n/sum(n))%>%
  filter(ccown=="Yes",!(is.na(pmttypeuse_a)))

fig38_ccown_noage<- merged_data %>% filter(idtag_diarycrosswave == 1)%>%
  count(pmttypeuse_a,ccown,wt=weight) %>% 
  group_by(pmttypeuse_a) %>%
  mutate(freq=n/sum(n))%>%
  filter(ccown=="Yes",!(is.na(pmttypeuse_a)))%>%
  mutate(agebin="All")

fig38<-rbind(fig40_bnpl_ccown, fig40_ccown_noage) %>% 
  ungroup() %>% 
  select(-ccown, -n) %>% 
  mutate(freq = 100*freq) %>% 
  setNames(c("Use BNPL", "agebin", "freq"))


## Figure 39: Average Number of BNPL Accounts Held ##
fig39_panel1 <- merged_data %>% 
  filter(idtag_diarycrosswave == 1) %>% 
  select(Respondent_ID, weight, agebin, starts_with("bnplaccount")) %>% 
  select(-bnplaccount_MINA) %>% 
  mutate(bnplaccount_8_other = ifelse(!is.na(bnplaccount_8_other) & stringr::str_detect(bnplaccount_8, "No"), "Yes", "No"),
         bnplaccount_9_other = ifelse(!is.na(bnplaccount_9_other) & stringr::str_detect(bnplaccount_9, "No"), "Yes", "No"),
         bnplaccount_10_other = ifelse(!is.na(bnplaccount_10_other) & stringr::str_detect(bnplaccount_10, "No"), "Yes", "No")) %>% 
  pivot_longer(., cols = names(.)[-c(1,2,3,length(names(.)))]) %>% 
  group_by(Respondent_ID, weight, bnplaccount_11, agebin) %>% 
  count(value) %>% 
  group_by(Respondent_ID) %>% 
  mutate(number = ifelse(bnplaccount_11 == "Yes", 0, n)) %>% 
  mutate(to_keep = case_when(
    n() == 1 ~ "keep",
    n() > 1 & value == "Yes" ~ "keep",
    T ~ "don't keep"
  )) %>% 
  filter(to_keep == "keep") %>% 
  group_by(agebin) %>%
  summarise(total = sum(number*weight),
            N = n()) %>% 
  ungroup() %>% 
  mutate(average = total/N, 
         `All respondents` = sum(total)/sum(N)) %>% 
  select(-total, -N) %>% 
  mutate(type = "All people")


fig39_panel2 <- merged_data %>% 
  filter(idtag_diarycrosswave == 1) %>% 
  select(Respondent_ID, weight, agebin, starts_with("bnplaccount")) %>% 
  select(-bnplaccount_MINA) %>% 
  mutate(bnplaccount_8_other = ifelse(!is.na(bnplaccount_8_other) & str_detect(bnplaccount_8, "No"), "Yes", "No"),
         bnplaccount_9_other = ifelse(!is.na(bnplaccount_9_other) & str_detect(bnplaccount_9, "No"), "Yes", "No"),
         bnplaccount_10_other = ifelse(!is.na(bnplaccount_10_other) & str_detect(bnplaccount_10, "No"), "Yes", "No")) %>% 
  pivot_longer(., cols = names(.)[-c(1,2,3,length(names(.)))]) %>% 
  group_by(Respondent_ID, weight, bnplaccount_11, agebin) %>% 
  count(value) %>% 
  group_by(Respondent_ID) %>% 
  mutate(number = ifelse(bnplaccount_11 == "Yes", 0, n)) %>% 
  filter(bnplaccount_11 == "No") %>% 
  mutate(to_keep = case_when(
    n() == 1 ~ "keep",
    n() > 1 & value == "Yes" ~ "keep",
    T ~ "don't keep"
  )) %>% 
  filter(to_keep == "keep") %>% 
  group_by(agebin) %>%
  summarise(total = sum(number*weight),
            N = n()) %>% 
  ungroup() %>% 
  mutate(average = total/N,
         `All respondents` = sum(total)/sum(N)) %>% 
  select(-N, -total) %>% 
  mutate(type = "Account holders")

fig39 <- rbind(fig39_panel1, fig39_panel2)


## Figure 40: Bank Transfers ##

fig40_age_graph <- merged_data  %>% 
  filter(paymeth_original != "Missing", !is.na(paymeth_original), payamt < 9999, payamt >= 0) %>% 
  filter(!(paypurp_crosswave %in% c("Cash deposit", "Transfers to a family member or friend", "Transfer within own accounts"))) %>%
  group_by(agebin, paymeth_original) %>% 
  summarise(val = sum(payamt*weight),
            vol = sum(weight)) %>% 
  group_by(agebin) %>% 
  mutate(val = val*100/sum(val),
         vol = vol*100/sum(vol)) %>% 
  filter(str_detect(paymeth_original, "[Tt]ransfer")) %>% 
  mutate(paymeth_crosswave = ifelse(str_detect(paymeth_original, "PayID"), "PayID", "BSB/Account")) %>% 
  group_by(agebin) %>% 
  mutate(share_vol = vol*100/sum(vol))



## Figure 41: Reasons for Not Using PayID ##
# Get heard data
fig41_heard_of <- merged_data %>% 
  filter(idtag_diarycrosswave == 1) %>% 
  select(Respondent_ID, weight, starts_with("pmttypeheard")) %>% 
  pivot_longer(., cols = names(.)[-c(1:2)]) %>% 
  filter(!str_detect(name, "MINA")) %>% 
  setNames(c("Respondent_ID", "weight", "type", "heard")) %>% 
  mutate(type = str_sub(type, -1, -1))

# Get long not used
fig41_not_used <- merged_data %>% 
  filter(idtag_diarycrosswave == 1) %>% 
  select(Respondent_ID, weight, starts_with("notused")) %>% 
  pivot_longer(., cols = names(.)[-c(1:2)]) %>% 
  filter(!str_detect(name, "MINA")) %>% 
  mutate(notusedreason = parse_number(name)) %>% 
  setNames(c("Respondent_ID", "weight", "type", "notused?", "notusedreason")) %>% 
  mutate(type = str_sub(type, -3, -3)) %>% 
  select(-weight)

# Join by type and respondent_id
fig41 <- left_join(fig41_heard_of, fig41_not_used, by = c("Respondent_ID", "type")) %>% 
  select(Respondent_ID, weight, type, heard, notusedreason, `notused?`) %>% 
  group_by(type, heard, notusedreason) %>% 
  count(`notused?`, wt = weight) %>% 
  mutate(type = case_when(
    type == "a" ~ "BNPL",
    type == "b" ~ "PayPal",
    type == "c" ~ "AliPay/WeChat",
    type == "d" ~ "Beem It",
    type == "e" ~ "In-app",
    type == "f" ~ "Tap and go",
    type == "g" ~ "PayID",
    type == "h" ~ "QR code",
    type == "i" ~ "Cryptocurrency"
  )) %>% 
  mutate(notusedreason = case_when(
    notusedreason == 1 ~ "What I have\nmeets my needs",
    notusedreason %in% c(2, 3) ~ "Don't know how to use/\ndon't have necessary tools",
    notusedreason %in% c(4, 5) ~ "Concerned about privacy\nand/or security",
    notusedreason == 6 ~ "Worried about keeping\ncontrol of spending",
    notusedreason == 7 ~ "Sellers don't\naccept method",
    T ~ "Other"
  )) %>% 
  filter(heard == "Yes", `notused?` == "Yes") %>% 
  group_by(type, notusedreason) %>% 
  summarise(n = sum(n)) %>% 
  group_by(type) %>% 
  mutate(perc = n*100/sum(n)) %>% 
  ungroup() %>% 
  select(type, notusedreason, perc)


## Figure 42: Person-to-person payments by channel ##
# 2022
d22 <- merged_data %>%
  filter(paypurp_crosswave == "Transfers to a family member or friend",
         !(paychan_crosswave %in% c("NA", "Missing")), payamt >= 0, payamt < 9999) %>%
  mutate(counttrans = 1) %>%
  group_by(paychan2019) %>%
  summarise(number = sum(weight*counttrans)) %>% 
  ungroup() %>% 
  mutate(perc = number*100/sum(number)) %>%
  mutate(channel = case_when(
    paychan2019 == "Offline" ~ "Offline",
    paychan2019 == "Online" ~ "Internet",
    paychan2019 == "App" ~ "Mobile",
    paychan2019 == "Other in person" ~ "Face-to-face",
    paychan2019 == "Tap device" ~ "Face-to-face"
  )) %>%
  group_by(channel) %>%
  summarise(perc = sum(perc)) %>%
  na.omit() %>%
  mutate(year = 2022)

# 2019
d19 <- data19 %>% 
  filter(paypurp == "Transfers to a family member or friend (17)", !(paychan %in% c("NA (-1)", "Missing (-5)")),
         as.numeric(payamt) < 9999, as.numeric(payamt) >= 0) %>%
  mutate(counttrans = 1) %>%
  group_by(paychan) %>%
  summarise(number = sum(as.numeric(weight)*counttrans)) %>%
  ungroup() %>%
  mutate(perc = number*100/sum(number)) %>%
  mutate(channel = case_when(
    paychan == "Offline (7)" ~ "Offline",
    paychan == "Online (6)" ~ "Internet",
    paychan == "App (5)" ~ "Mobile",
    paychan == "Other in person (4)" ~ "Face-to-face",
    paychan == "Tap card (1)" ~ "Face-to-face"
  )) %>%
  group_by(channel) %>%
  summarise(perc = sum(perc)) %>%
  mutate(year = 2019)



# 2016
d16 <- data16 %>% 
  filter(paypurp == "Transfers to a family member or friend (17)", !(paychan %in% c("NA (-1)", "Missing (-5)")),
         as.numeric(payamt) < 9999, as.numeric(payamt) >= 0) %>%
  mutate(counttrans = 1) %>%
  group_by(paychan) %>%
  summarise(number = sum(as.numeric(w1_noint)*counttrans)) %>%
  ungroup() %>%
  mutate(perc = number*100/sum(number)) %>%
  mutate(channel = case_when(
    paychan == "Mail (5)" ~ "Offline",
    paychan == "Internet(PC/tablet) (2)" ~ "Internet",
    paychan == "Internet(Smartphone) (3)" ~ "Mobile",
    paychan == "In person (1)" ~ "Face-to-face",
  )) %>%
  na.omit() %>%
  select(-paychan, -number) %>%
  mutate(year = 2016)


# 2013
d13 <- data13 %>% 
  filter(paypurp == "Transfers to a family member or friend", !(paychan %in% c("NA", "Missing")),
         as.numeric(payamt) < 9999, as.numeric(payamt) >= 0) %>%
  mutate(counttrans = 1) %>%
  group_by(paychan) %>%
  summarise(number = sum(as.numeric(weight1)*counttrans)) %>%
  ungroup() %>%
  mutate(perc = number*100/sum(number)) %>%
  mutate(channel = case_when(
    paychan == "Mail" ~ "Offline",
    paychan == "Internet(PC/tablet)" ~ "Internet",
    paychan == "Smartphone/SMS" ~ "Mobile",
    paychan == "In person" ~ "Face-to-face",
    paychan == "Telephone" ~ "Offline"
  )) %>%
  group_by(channel) %>%
  summarise(perc = sum(perc)) %>%
  mutate(year = 2013)


fig42 <- rbind(d13, d16, d19, d22)

##
